import { Knex } from 'knex';
import { Collection } from './Collection';
import { CollectionRepository } from './CollectionRepository';
import { CollectionPost } from './CollectionPost';
import { PostEditedEvent } from './events/PostEditedEvent';
interface SlugService {
    generate(desired: string, options: {
        transaction: Knex.Transaction;
    }): Promise<string>;
}
type CollectionsServiceDeps = {
    collectionsRepository: CollectionRepository;
    postsRepository: PostsRepository;
    slugService: SlugService;
    DomainEvents: {
        subscribe: (event: any, handler: (e: any) => void) => void;
    };
};
type CollectionPostDTO = {
    id: string;
    sort_order: number;
};
type CollectionPostListItemDTO = {
    id: string;
    url: string;
    slug: string;
    title: string;
    featured: boolean;
    featured_image?: string;
    created_at: Date;
    updated_at: Date;
    published_at: Date;
    tags: Array<{
        slug: string;
    }>;
};
type ManualCollection = {
    title: string;
    type: 'manual';
    slug?: string;
    description?: string;
    feature_image?: string;
    filter?: null;
    deletable?: boolean;
};
type AutomaticCollection = {
    title: string;
    type: 'automatic';
    filter: string;
    slug?: string;
    description?: string;
    feature_image?: string;
    deletable?: boolean;
};
type CollectionInputDTO = ManualCollection | AutomaticCollection;
type CollectionDTO = {
    id: string;
    title: string | null;
    slug: string;
    description: string | null;
    feature_image: string | null;
    type: 'manual' | 'automatic';
    filter: string | null;
    created_at: Date;
    updated_at: Date | null;
    posts: CollectionPostDTO[];
};
type QueryOptions = {
    filter?: string;
    include?: string;
    page?: number;
    limit?: number;
    transaction?: Knex.Transaction;
};
interface PostsRepository {
    getAll(options: QueryOptions): Promise<CollectionPost[]>;
    getAllIds(options?: {
        transaction: Knex.Transaction;
    }): Promise<string[]>;
}
export declare class CollectionsService {
    private collectionsRepository;
    private postsRepository;
    private DomainEvents;
    private uniqueChecker;
    private slugService;
    constructor(deps: CollectionsServiceDeps);
    private toDTO;
    private fromDTO;
    /**
     * @description Subscribes to Domain events to update collections when posts are added, updated or deleted
     */
    subscribeToEvents(): void;
    private hasPostEditRelevantChanges;
    updateAllAutomaticCollections(): Promise<void>;
    createCollection(data: CollectionInputDTO): Promise<CollectionDTO>;
    addPostToCollection(collectionId: string, post: CollectionPostListItemDTO): Promise<CollectionDTO | null>;
    removePostFromAllCollections(postId: string): Promise<any>;
    removePostsFromAllCollections(postIds: string[]): Promise<any>;
    private addPostToMatchingCollections;
    updatePostInMatchingCollections(postEdit: PostEditedEvent['data']): Promise<any>;
    updateUnpublishedPosts(postIds: string[]): Promise<any>;
    updateFeaturedPosts(postIds: string[]): Promise<any>;
    updatePostsInCollections(postIds: string[], collections: Collection[], transaction: Knex.Transaction): Promise<void>;
    edit(data: any): Promise<CollectionDTO | null>;
    getById(id: string, options?: {
        transaction: Knex.Transaction;
    }): Promise<CollectionDTO | null>;
    getBySlug(slug: string, options?: {
        transaction: Knex.Transaction;
    }): Promise<CollectionDTO | null>;
    getAll(options?: QueryOptions): Promise<{
        data: CollectionDTO[];
        meta: any;
    }>;
    getCollectionsForPost(postId: string): Promise<CollectionDTO[]>;
    destroy(id: string): Promise<Collection | null>;
    removePostFromCollection(id: string, postId: string): Promise<CollectionDTO | null>;
}
export {};
